<?php
// /notifications_ajax.php

// --- GESTIÓN DE SESIÓN Y DEPENDENCIAS ---
session_start();
if (!isset($_SESSION['user_id'])) {
    http_response_code(401); // Unauthorized
    echo json_encode(['status' => 'error', 'message' => 'Acceso no autorizado.']);
    exit;
}
// Solo fisios y superadmins pueden usar esto
if (!in_array($_SESSION['user_rol'], ['superadmin', 'fisio'])) {
     http_response_code(403); // Forbidden
     echo json_encode(['status' => 'error', 'message' => 'Acceso denegado para este rol.']);
     exit;
}

require_once 'db.php'; // Asegúrate de que $db esté disponible
$user_id = $_SESSION['user_id'];
$action = $_REQUEST['action'] ?? '';

// --- RESPUESTA JSON ---
header('Content-Type: application/json');

try {
    switch ($action) {

        // --- MODIFICADO: 'get_notifications' con 3 filtros ---
        case 'get_notifications':
            $filter = $_GET['filter'] ?? 'unread'; // 'unread' (default), 'read', o 'all'

            // 1. Contar SIEMPRE las no leídas (para el badge global)
            $stmt_count = $db->prepare("SELECT COUNT(id) FROM notificaciones WHERE fisio_id = :fisio_id AND leido_en IS NULL");
            $stmt_count->execute([':fisio_id' => $user_id]);
            $unread_count = $stmt_count->fetchColumn() ?: 0;
            $stmt_count->closeCursor();

            // 2. Construir la consulta de obtención basada en el filtro
            $limit = 30; // Un límite razonable para la vista "Totes"
            $where_sql = "fisio_id = :fisio_id";

            if ($filter === 'unread') {
                $where_sql .= " AND leido_en IS NULL";
            } elseif ($filter === 'read') {
                $where_sql .= " AND leido_en IS NOT NULL";
            }
            // Si $filter es 'all', no añadimos más condiciones

            $stmt_fetch = $db->prepare("
                SELECT id, tipo_evento, mensaje, url_destino, fecha_creacion, leido_en
                FROM notificaciones
                WHERE $where_sql
                ORDER BY fecha_creacion DESC
                LIMIT :limit
            ");
            $stmt_fetch->bindValue(':fisio_id', $user_id, PDO::PARAM_INT);
            $stmt_fetch->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt_fetch->execute();
            $notifications = $stmt_fetch->fetchAll(PDO::FETCH_ASSOC);
            $stmt_fetch->closeCursor();

            echo json_encode([
                'status' => 'success',
                'count' => (int)$unread_count, // El recuento de NO LEÍDAS
                'notifications' => $notifications // La lista filtrada
            ]);
            break;

        // --- SIN CAMBIOS: Sigue marcando como leído ---
        case 'delete_notification':
            $notification_id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);

            if (!$notification_id) {
                throw new Exception('ID de notificación no válido.');
            }

            $stmt_mark_read = $db->prepare(
                            "UPDATE notificaciones
                             SET leido_en = NOW()
                             WHERE id = :id AND fisio_id = :fisio_id AND leido_en IS NULL"
                        );

            $stmt_mark_read->execute([
                ':id' => $notification_id,
                ':fisio_id' => $user_id
            ]);

            if ($stmt_mark_read->rowCount() > 0) {
                 echo json_encode(['status' => 'success', 'message' => 'Notificación marcada como leída.']);
            } else {
                 echo json_encode(['status' => 'success', 'message' => 'No se marcó nada (quizás ya leída).']);
            }
            break;

        // --- SIN CAMBIOS: Sigue marcando TODAS como leídas ---
        // (La acción se llama 'delete_all_unread' pero MARCA como leído)
        case 'delete_all_unread':

        $stmt_mark_all = $db->prepare(
              "UPDATE notificaciones
               SET leido_en = NOW()
               WHERE fisio_id = :fisio_id AND leido_en IS NULL"
          );

            $stmt_mark_all->execute([':fisio_id' => $user_id]);
            $marked_count = $stmt_mark_all->rowCount();

            echo json_encode([
                'status' => 'success',
                'message' => "Se marcaron $marked_count notificaciones como leídas."
            ]);
            break;

        // --- NUEVA ACCIÓN: Borrado permanente de una notificación ---
        case 'permanently_delete_notification':
            $notification_id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);

            if (!$notification_id) {
                throw new Exception('ID de notificación no válido.');
            }

            $stmt_delete = $db->prepare(
                "DELETE FROM notificaciones
                 WHERE id = :id AND fisio_id = :fisio_id"
            );
            $stmt_delete->execute([
                ':id' => $notification_id,
                ':fisio_id' => $user_id
            ]);

            if ($stmt_delete->rowCount() > 0) {
                 echo json_encode(['status' => 'success', 'message' => 'Notificación eliminada permanentemente.']);
            } else {
                 echo json_encode(['status' => 'error', 'message' => 'No se pudo eliminar la notificación.']);
            }
            break;

        // --- NUEVA ACCIÓN: Borrado permanente de TODAS las LEÍDAS ---
        case 'delete_all_read':
            $stmt_delete_all = $db->prepare(
                "DELETE FROM notificaciones
                 WHERE fisio_id = :fisio_id AND leido_en IS NOT NULL"
            );
            $stmt_delete_all->execute([':fisio_id' => $user_id]);
            $deleted_count = $stmt_delete_all->rowCount();

             echo json_encode([
                'status' => 'success',
                'message' => "S'han eliminat $deleted_count notificacions llegides."
            ]);
            break;


        default:
            throw new Exception('Acción no válida solicitada.');
    }

} catch (PDOException $e) {
    error_log("Error de base de datos en notifications_ajax.php (Action: $action): " . $e->getMessage());
    http_response_code(500); // Internal Server Error
    echo json_encode(['status' => 'error', 'message' => 'Error interno del servidor al procesar la solicitud.']);
} catch (Exception $e) {
    http_response_code(400); // Bad Request
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}

exit; // Terminar ejecución
?>
