<?php
// /patient_report.php
// CORREGIDO: Sintaxis de CONCAT() para MySQL

// --- 1. GESTIÓN DE SESIÓN Y DEPENDENCIAS ---
session_start();
if (!isset($_SESSION['user_id']) || $_SESSION['user_rol'] !== 'paciente') {
    header('Location: index.php');
    exit;
}
require '../db.php';
$patient_id = $_SESSION['user_id'];
$page_title = "Informe de Pacient";

// --- 2. RECOPILACIÓN DE DATOS ---

// Datos del Paciente
$stmt_pacient = $db->prepare("SELECT nombre, apellido, email, telefono FROM cuentas WHERE id = ?");
$stmt_pacient->execute([$patient_id]);
$pacient = $stmt_pacient->fetch(PDO::FETCH_ASSOC);

// Todos los Tratamientos (luego los separamos)
// --- ★★★ INICIO CORRECCIÓN MYSQL ★★★ ---
// (GROUP_CONCAT(c.nombre || ' ' || c.apellido, ', ') -> CONCAT(c.nombre, ' ', c.apellido) SEPARATOR ', ')
$stmt_treatments = $db->prepare(
    "SELECT t.id, t.title, t.status, t.start_date, t.end_date, t.diagnostico, t.anamnesis,
           (SELECT GROUP_CONCAT(CONCAT(c.nombre, ' ', c.apellido) SEPARATOR ', ')
            FROM tratamiento_fisios_asignados tfa
            JOIN cuentas c ON tfa.fisio_id = c.id
            WHERE tfa.tratamiento_id = t.id) as fisios_asignados
     FROM tratamientos t
     WHERE t.paciente_id = :patient_id AND t.is_protocol = 0
     ORDER BY t.start_date DESC"
);
// --- ★★★ FIN CORRECCIÓN MYSQL ★★★ ---
$stmt_treatments->execute([':patient_id' => $patient_id]);
$all_treatments = $stmt_treatments->fetchAll(PDO::FETCH_ASSOC);

$active_treatments = [];
$history_treatments = [];
foreach ($all_treatments as $t) {
    if (in_array($t['status'], ['En curs', 'Programat'])) {
        $active_treatments[] = $t;
    } else {
        $history_treatments[] = $t;
    }
}

// Todo el Historial de Evolución
$stmt_evo = $db->prepare(
    "SELECT
        t.title as treatment_title,
        e.title as exercise_title,
        ev.fecha_realizacion,
        ev.dolor_percibido,
        ev.esfuerzo_percibido,
        ev.comentarios
     FROM tratamiento_evolucion ev
     JOIN tratamiento_ejercicios te ON ev.tratamiento_ejercicio_id = te.id
     JOIN tratamientos t ON te.tratamiento_id = t.id
     JOIN ejercicios e ON te.ejercicio_id = e.id
     WHERE t.paciente_id = :patient_id
     ORDER BY ev.fecha_realizacion DESC, t.title, e.title"
);
$stmt_evo->execute([':patient_id' => $patient_id]);
$all_evolution = $stmt_evo->fetchAll(PDO::FETCH_ASSOC);

// Agrupar evolución por fecha
$evolution_by_date = [];
foreach ($all_evolution as $evo) {
    $evolution_by_date[$evo['fecha_realizacion']][] = $evo;
}

// Función simple para formatear fechas
function formatDate($dateStr) {
    if (empty($dateStr)) return 'N/A';
    try {
        return (new DateTime($dateStr))->format('d/m/Y');
    } catch (Exception $e) {
        return 'Data invàlida';
    }
}
?>
<!DOCTYPE html>
<html lang="ca">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
    <style>
        :root {
            --bs-body-font-family: "Segoe UI", "Helvetica Neue", Arial, sans-serif;
        }

        body {
            background-color: #e9ecef; /* Fondo gris claro para la vista de pantalla */
            -webkit-print-color-adjust: exact !important; /* Forzar colores en impresión (Chrome) */
            color-adjust: exact !important; /* Forzar colores en impresión (Firefox) */
        }

        .report-container {
            max-width: 800px;
            margin: 20px auto;
            padding: 40px;
            background-color: #ffffff;
            border: 1px solid #dee2e6;
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, .075);
            font-family: var(--bs-body-font-family);
        }

        .report-header {
            text-align: center;
            border-bottom: 2px solid #000;
            padding-bottom: 15px;
            margin-bottom: 30px;
        }
        .report-header h1 {
            font-weight: 300;
            margin-bottom: 0;
        }
        .report-header h4 {
            font-weight: 700;
            color: #31a3a3; /* Color corporativo */
        }

        .section-title {
            font-weight: 600;
            border-bottom: 1px solid #dee2e6;
            padding-bottom: 8px;
            margin-top: 25px;
            margin-bottom: 15px;
            color: #31a3a3;
        }

        .treatment-card {
            border: 1px solid #dee2e6;
            padding: 15px;
            margin-bottom: 15px;
            border-radius: 5px;
            background-color: #f8f9fa;
        }
        .treatment-card h5 {
            font-weight: 600;
        }

        .evo-table {
            font-size: 0.85rem;
        }
        .evo-table th {
            background-color: #f8f9fa;
        }

        .data-list dt {
            font-weight: 600;
        }
        .data-list dd {
            margin-bottom: 0.5rem;
        }

        .print-button-container {
            text-align: center;
            margin-bottom: 20px;
        }

        /* --- Estilos de Impresión --- */
        @media print {
            body {
                background-color: #ffffff;
                margin: 0;
                font-size: 10pt; /* Tamaño más pequeño para impresión */
            }

            .no-print {
                display: none !important; /* Ocultar botón de imprimir y contenedor gris */
            }

            .report-container {
                width: 100%;
                max-width: 100%;
                margin: 0;
                padding: 10mm; /* Márgenes de la página */
                border: none;
                box-shadow: none;
            }

            .page-break {
                page-break-after: always; /* Forzar salto de página */
            }

            .treatment-card {
                background-color: #f8f9fa !important; /* Forzar fondo en impresión */
                border: 1px solid #ced4da !important;
            }

            .evo-table th {
                background-color: #f8f9fa !important;
            }

            .evo-table {
                font-size: 9pt; /* Aún más pequeño en impresión si es necesario */
            }

            a {
                text-decoration: none;
                color: #000 !important;
            }
        }
    </style>
</head>
<body>

    <div class="print-button-container no-print">
        <div class="report-container" style="padding: 20px; box-shadow: none;">
            <button id="print-btn" class="btn btn-primary w-100">
                <i class="bi bi-printer me-2"></i>Imprimir Informe
            </button>
        </div>
    </div>

    <div class="report-container" id="report-content">
        <header class="report-header">
            <h4>AVANT ONLINE</h4>
            <h1>Informe de Progrés del Pacient</h1>
            <p class="text-muted mb-0">Generat el: <?= (new DateTime())->format('d/m/Y H:i') ?></p>
        </header>

        <main>
            <section id="patient-data">
                <h3 class="section-title">Dades del Pacient</h3>
                <dl class="row data-list">
                    <dt class="col-sm-3">Nom Complet</dt>
                    <dd class="col-sm-9"><?= htmlspecialchars($pacient['nombre'] ?? '') ?> <?= htmlspecialchars($pacient['apellido'] ?? '') ?></dd>

                    <dt class="col-sm-3">Email</dt>
                    <dd class="col-sm-9"><?= htmlspecialchars($pacient['email'] ?? '') ?></dd>

                    <dt class="col-sm-3">Telèfon</dt>
                    <dd class="col-sm-9"><?= htmlspecialchars($pacient['telefono'] ?? 'No especificat') ?></dd>
                </dl>
            </section>

            <section id="active-treatments">
                <h3 class="section-title">Tractaments Actius / Programats</h3>
                <?php if (empty($active_treatments)): ?>
                    <p class="text-muted">No hi ha tractaments actius actualment.</p>
                <?php else: ?>
                    <?php foreach ($active_treatments as $t): ?>
                    <div class="treatment-card">
                        <h5><?= htmlspecialchars($t['title']) ?></h5>
                        <dl class="row data-list mb-0 small">
                            <dt class="col-sm-3">Estat</dt>
                            <dd class="col-sm-9"><?= htmlspecialchars($t['status']) ?></dd>
                            <dt class="col-sm-3">Rang de Dates</dt>
                            <dd class="col-sm-9"><?= formatDate($t['start_date']) ?> - <?= formatDate($t['end_date']) ?></dd>
                            <dt class="col-sm-3">Fisios</dt>
                            <dd class="col-sm-9"><?= htmlspecialchars($t['fisios_asignados'] ?? 'N/A') ?></dd>
                            <dt class="col-sm-3">Diagnòstic</dt>
                            <dd class="col-sm-9"><?= htmlspecialchars($t['diagnostico'] ?? 'No especificat') ?></dd>
                        </dl>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </section>

            <section id="history-treatments">
                <h3 class="section-title">Historial de Tractaments</h3>
                <?php if (empty($history_treatments)): ?>
                    <p class="text-muted">No hi ha tractaments en l'historial.</p>
                <?php else: ?>
                    <?php foreach ($history_treatments as $t): ?>
                    <div class="treatment-card">
                        <h5><?= htmlspecialchars($t['title']) ?></h5>
                        <dl class="row data-list mb-0 small">
                            <dt class="col-sm-3">Estat</dt>
                            <dd class="col-sm-9"><?= htmlspecialchars($t['status']) ?></dd>
                            <dt class="col-sm-3">Rang de Dates</dt>
                            <dd class="col-sm-9"><?= formatDate($t['start_date']) ?> - <?= formatDate($t['end_date']) ?></dd>
                            <dt class="col-sm-3">Fisios</dt>
                            <dd class="col-sm-9"><?= htmlspecialchars($t['fisios_asignados'] ?? 'N/A') ?></dd>
                        </dl>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </section>

            <div class="page-break"></div>

            <section id="evolution-log">
                <h3 class="section-title">Registre d'Evolució</h3>
                <?php if (empty($evolution_by_date)): ?>
                    <p class="text-muted">No s'han trobat registres d'evolució.</p>
                <?php else: ?>
                    <?php foreach ($evolution_by_date as $date => $records): ?>
                        <h5 class="mt-4">Registre del <?= formatDate($date) ?></h5>
                        <table class="table table-sm table-bordered table-striped evo-table">
                            <thead class="table-light">
                                <tr>
                                    <th>Tractament</th>
                                    <th>Exercici</th>
                                    <th>Dolor (0-5)</th>
                                    <th>Esforç (0-5)</th>
                                    <th>Comentaris</th>
                                </tr>
                            </thead>
                            <tbody>
                            <?php foreach ($records as $rec): ?>
                                <tr>
                                    <td><?= htmlspecialchars($rec['treatment_title']) ?></td>
                                    <td><?= htmlspecialchars($rec['exercise_title']) ?></td>
                                    <td class="text-center"><?= htmlspecialchars($rec['dolor_percibido']) ?></td>
                                    <td class="text-center"><?= htmlspecialchars($rec['esfuerzo_percibido']) ?></td>
                                    <td><?= htmlspecialchars($rec['comentarios'] ?? 'Cap') ?></td>
                                </tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endforeach; ?>
                <?php endif; ?>
            </section>

        </main>
    </div>

    <script>
        // Script simple para activar la impresión
        document.getElementById('print-btn').addEventListener('click', function() {
            window.print();
        });
    </script>
</body>
</html>
