<?php
// /patient_ajax.php - Gestiona todas las peticiones AJAX para el dashboard del paciente
// --- MODIFICADO: AHORA TAMBIÉN GESTIONA LAS PETICIONES DE fitxa_pacient.php ---
// v2: Logs B1, B2, B3 centralizados en logger.php

// --- 1. GESTIÓN DE SESIÓN Y DEPENDENCIAS ---
session_start();
header('Content-Type: application/json'); // Asegurar respuesta JSON

// --- MODIFICACIÓN DE SEGURIDAD ---
// Validar que HAYA una sesión. Ya no bloquea a fisios.
if (!isset($_SESSION['user_id'])) {
    http_response_code(401); // Unauthorized
    echo json_encode(['status' => 'error', 'message' => 'Accés denegat. Sessió no iniciada.']);
    exit;
}
// --- FIN MODIFICACIÓN ---

require_once 'db.php'; // Necesario para la conexión $db
require_once 'patient_functions.php'; // Incluir el archivo con TODA la lógica
require_once 'avatar_helpers.php'; // ★ INCLUIR EL NUEVO HELPER DE AVATAR ★
require_once 'logger.php'; // <-- ★★★ AÑADIDO PARA LOGS B1, B2, B3 ★★★

// ===== Variables globales para los 'cases' =====
$action = $_REQUEST['action'] ?? $_GET['action'] ?? '';
$user_rol = $_SESSION['user_rol']; // Rol del usuario logueado

// Distinguir si el usuario logueado es el paciente o un fisio
if ($user_rol === 'paciente') {
    $patient_id = $_SESSION['user_id']; // El paciente es el usuario
    $current_fisio_id = 0; // No aplicable
} else {
    // Es un fisio/admin. El ID del fisio es la sesión.
    $current_fisio_id = $_SESSION['user_id'];
    // El ID del paciente viene en la petición AJAX (de fitxa_pacient.php)
    $patient_id = (int)($_REQUEST['pacient_id'] ?? $_REQUEST['id'] ?? 0);
}
// =============================================

// --- 2. GESTOR DE PETICIONES AJAX (Controlador) ---
try {
    $response = []; // Inicializar la respuesta

    switch($action) {

        // =======================================================
        // --- ACCIONES DEL PACIENTE (patient_dashboard.php) ---
        // =======================================================

        case 'update_last_seen':
            // Esta acción la pueden llamar ambos roles
            if ($user_rol === 'paciente') {
                $response = updatePatientLastSeen($db, $patient_id);
            } else {
                // (Los fisios usan 'reports_ajax.php' para esto, pero no daña si se queda aquí)
                $response = ['status' => 'success', 'message' => 'Fisio last_seen updated (ignorado).'];
            }
            break;

        case 'get_active_treatment_data':
            $response = getActiveTreatmentData($db, $patient_id);
            break;

        case 'check_history_exists':
            $response = checkPatientHistoryExists($db, $patient_id);
            break;

        case 'get_treatment_details':
            $treatment_id = (int)($_GET['treatment_id'] ?? 0);
            $response = getPatientTreatmentDetails($db, $patient_id, $treatment_id);
            break;

        case 'get_full_evolution': // Usado por el dashboard de paciente
            $treatment_id = (int)($_GET['treatment_id'] ?? 0);
            $exercise_id_filter = (int)($_GET['exercise_id'] ?? 0);
            $response = getPatientFullEvolution($db, $patient_id, $treatment_id, $exercise_id_filter);
            break;

            // B2: Guardar Feedback del Paciente
            case 'save_exercise_feedback':
                $response = savePatientExerciseFeedback($db, $patient_id, $_POST);

                // --- ★★★ INICIO LOG B2 (Feedback Guardado) ★★★ ---
                if (isset($response['status']) && $response['status'] === 'success') {
                    $tratamiento_ejercicio_id = $_POST['tratamiento_ejercicio_id'] ?? null;
                    $log_details_b2 = "Paciente (ID: $patient_id) guardó feedback para el ejercicio (ID tratamiento_ejercicio: $tratamiento_ejercicio_id).";
                    registrar_log($db, 'FEEDBACK_SAVED', $patient_id, $_SESSION['user_nombre'], $log_details_b2, $patient_id, $tratamiento_ejercicio_id);
                }
                // --- ★★★ FIN LOG B2 ★★★ ---
                break;

                // B3: Borrar Feedback del Paciente
                case 'delete_exercise_feedback':
                    $response = deletePatientExerciseFeedback($db, $patient_id, $_POST);

                    // --- ★★★ INICIO LOG B3 (Feedback Borrado) ★★★ ---
                    if (isset($response['status']) && $response['status'] === 'success') {
                        $feedback_id = $_POST['feedback_id'] ?? null;
                        $log_details_b3 = "Paciente (ID: $patient_id) borró feedback (ID feedback: $feedback_id).";
                        registrar_log($db, 'FEEDBACK_DELETED', $patient_id, $_SESSION['user_nombre'], $log_details_b3, $patient_id, $feedback_id);
                    }
                    // --- ★★★ FIN LOG B3 ★★★ ---
                    break;

            // --- ★ INICIO MODIFICACIÓN: 'get_patient_data' ★ ---
                    case 'get_patient_data':
                         // --- ★ INICIO MODIFICACIÓN (Bloqueo de Perfil en Suplantación) ★ ---
                         if (isset($_SESSION['admin_origen_id'])) {
                            throw new Exception('Accés denegat durant la suplantació.');
                         }
                         // --- ★ FIN MODIFICACIÓN ★ ---

                         // (b) Reemplazar llamada a función por consulta directa
                         if ($user_rol !== 'paciente') {
                             throw new Exception('Accés denegat.');
                         }
                         $stmt = $db->prepare("SELECT nombre, apellido, email, telefono, avatar FROM cuentas WHERE id = ?");
                         $stmt->execute([$patient_id]);
                         $data = $stmt->fetch(PDO::FETCH_ASSOC);

                         if ($data) {
                             $response = ['status' => 'success', 'data' => $data];
                         } else {
                             throw new Exception('No s\'han trobat les dades del pacient.');
                         }
                         break;
                    // --- ★ FIN MODIFICACIÓN ★ ---

                    // --- ★ INICIO MODIFICACIÓN: 'update_personal_data' ★ ---
                            case 'update_personal_data':
                                // --- ★ INICIO MODIFICACIÓN (Bloqueo de Perfil en Suplantación) ★ ---
                                if (isset($_SESSION['admin_origen_id'])) {
                                   throw new Exception('Accés denegat durant la suplantació.');
                                }
                                // --- ★ FIN MODIFICACIÓN ★ ---

                                // (c) Reemplazar llamada a función por el helper de avatar
                                if ($user_rol !== 'paciente') {
                                     throw new Exception('Accés denegat.');
                                }
                                // (c) Llamar al helper. Este se encarga de todo:
                                // validar texto, validar imagen, borrar/mover archivo, actualizar BBDD y actualizar SESSió.
                                $response = handleAvatarUpload($db, $patient_id, $_POST, $_FILES, 'paciente');
                                break;
                            // --- ★ FIN MODIFICACIÓN ★ ---

                            case 'update_password':
                                        // --- ★ INICIO MODIFICACIÓN (Bloqueo de Perfil en Suplantación) ★ ---
                                        if (isset($_SESSION['admin_origen_id'])) {
                                           throw new Exception('Accés denegat durant la suplantació.');
                                        }
                                        // --- ★ FIN MODIFICACIÓN ★ ---

                                        $response = updatePatientPassword($db, $patient_id, $_POST);

                                        // --- ★★★ INICIO LOG B1 (Cambio Contraseña - Paciente) ★★★ ---
                                        if (isset($response['status']) && $response['status'] === 'success') {
                                            $log_details_b1 = "Paciente (ID: $patient_id) ha cambiado su propia contraseña desde 'Mi Perfil'.";
                                            registrar_log($db, 'PASSWORD_CHANGED_SELF', $patient_id, $_SESSION['user_nombre'], $log_details_b1, $patient_id, null);
                                        }
                                        // --- ★★★ FIN LOG B1 ★★★ ---
                                        break;

        case 'get_treatment_history':
            $response = getPatientTreatmentHistory($db, $patient_id);
            break;

        case 'get_unread_message_count':
            $response = getPatientUnreadMessageCount($db, $patient_id);
            break;

        case 'get_chat_messages':
            $fisio_id = (int)($_GET['fisio_id'] ?? 0);
            $response = getPatientChatMessages($db, $patient_id, $fisio_id);
            break;

        case 'send_chat_message':
            $response = sendPatientChatMessage($db, $patient_id, $_POST);

            // --- ★★★ INICIO LOG CHAT (Paciente) ★★★ ---
            if (isset($response['status']) && $response['status'] === 'success') {
                $target_fisio_id = $_POST['to_id'] ?? null; // El fisio ID ve en 'to_id'
                $message_id = $response['message_id'] ?? null;
                $log_details_chat = "Paciente (ID: $patient_id) envió mensaje a Fisio (ID: $target_fisio_id)";
                // El actor es el paciente, el target es el fisio
                registrar_log($db, 'CHAT_MESSAGE_SENT', $patient_id, $_SESSION['user_nombre'], $log_details_chat, $target_fisio_id, $message_id);
            }
            // --- ★★★ FIN LOG CHAT ★★★ ---

            break;

        case 'upload_chat_image':
            // Pasar $_POST y $_FILES a la función
            $response = uploadPatientChatImage($db, $patient_id, $_POST, $_FILES);
            break;

        case 'check_chat_updates':
            $fisio_id = (int)($_GET['fisio_id'] ?? 0);
            $response = checkPatientChatUpdates($db, $patient_id, $fisio_id);
            break;

        case 'delete_chat_message':
            $response = deletePatientChatMessage($db, $patient_id, $_POST);
            break;

        case 'delete_conversation':
            $response = deletePatientConversation($db, $patient_id, $_POST);
            break;

        case 'get_my_announcements':
             $response = getPatientAnnouncements($db, $patient_id, $user_rol);
             break;

        case 'mark_announcement_read':
            $response = markPatientAnnouncementRead($db, $patient_id, $_POST);
            break;

        // =======================================================
        // --- NUEVAS ACCIONES DEL FISIO (fitxa_pacient.php) ---
        // =======================================================

        case 'get_fisio_dependencies':
            // Esta acción es solo para fisios/admins
            if ($user_rol === 'paciente') { throw new Exception('Accés denegat.'); }
            $response = getFisioDependencies($db);
            break;

        case 'get_patient_treatments':
            if ($user_rol === 'paciente') { throw new Exception('Accés denegat.'); }
            if ($patient_id === 0) { throw new Exception('ID de pacient no proporcionat.'); }
            $response = getPatientTreatments($db, $patient_id, $current_fisio_id, $user_rol);
            break;

        case 'get_protocol_details_for_assignment':
            if ($user_rol === 'paciente') { throw new Exception('Accés denegat.'); }
            $protocol_id = (int)($_GET['protocol_id'] ?? 0);
            $response = getProtocolDetailsForAssignment($db, $protocol_id);
            break;

        case 'get_evolution_overview':
            if ($user_rol === 'paciente') { throw new Exception('Accés denegat.'); }
            if ($patient_id === 0) { throw new Exception('ID de pacient no proporcionat.'); }
            $treatment_id = (int)($_GET['treatment_id'] ?? 0);
            $response = getEvolutionOverview($db, $patient_id, $treatment_id);
            break;

        case 'get_full_evolution_data': // Usado por el modal de evolución del fisio
            if ($user_rol === 'paciente') { throw new Exception('Accés denegat.'); }
            if ($patient_id === 0) { throw new Exception('ID de pacient no proporcionat.'); }
            $treatment_id = (int)($_GET['treatment_id'] ?? 0);
            $exercise_id_filter = (int)($_GET['exercise_id'] ?? 0);
            $response = getPatientFullEvolutionData($db, $patient_id, $treatment_id, $exercise_id_filter, $current_fisio_id, $user_rol);
            break;

        default:
            throw new Exception('Acció AJAX no reconeguda.');
    }

    // Enviar la respuesta JSON al final
    echo json_encode($response);

} catch (Exception $e) {
    // Determinar código de error (403 si es denegado, 400 si es malo, 500 si es servidor)
    $errorCode = 400; // Bad Request por defecto
    if ($e->getMessage() === 'Accés denegat.') {
        $errorCode = 403; // Forbidden
    } elseif ($e instanceof PDOException) {
        $errorCode = 500; // Internal Server Error
    }

    http_response_code($errorCode);

    // Registrar el error detallado para depuración interna
    error_log("Error en patient_ajax.php (Action: $action, UserRole: $user_rol, PatientID: $patient_id, FisioID: $current_fisio_id): " . $e->getMessage());
    // Devolver un mensaje genérico al cliente
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
exit; // Asegurarse de que el script termina aquí
?>
