<?php
// /admin_logs.php - Visor del Registro de Auditoría (Corregido para MySQL + Mejoras Visuales)
// VERSIÓN v9 - Logs de las Series A, B, y C (Incluyendo Chat Logs)

// --- 1. GESTIÓN DE SESIÓN Y PERMISOS ---
session_start();

// ★★★ GUARDIA DE SEGURIDAD ★★★
// Solo accesible por Superadmins
if (!isset($_SESSION['user_id']) || $_SESSION['user_rol'] !== 'superadmin') {
    header('Location: dashboard.php');
    exit;
}

require_once 'db.php';
$page_title = "Registro de Auditoría";

// --- MATRIZ DE CATEGORÍAS Y DESCRIPCIONES (en Castellano) ---
$log_categories = [
    'A' => [
        'A1' => ['action' => 'LOGIN_SUCCESS', 'desc' => 'Evidencia de acceso de usuario correcto.'],
        'A2' => ['action' => 'LOGIN_FAIL', 'desc' => 'Intento de acceso no autorizado (Intrusión).'],
        'A3' => ['action' => 'IMPERSONATE_START', 'desc' => 'CRÍTICO: Inicio de control administrativo (Suplantación).'],
        'A4' => ['action' => 'IMPERSONATE_STOP', 'desc' => 'Finalización de control administrativo (Suplantación).'],
        'A5' => ['action' => 'BACKUP_RESTORED', 'desc' => 'Cambio masivo de datos (Restauración de BD).'],
        'A6' => ['action' => 'BACKUP_DELETED', 'desc' => 'Destrucción de un archivo de seguridad (Copia de seguridad).'],
        'A7' => ['action' => 'PURGE_EVOLUTIONS', 'desc' => 'Purga de registros obsoletos (Cumplimiento RGPD).'],
        'A8' => ['action' => 'BACKUP_CREATED', 'desc' => 'Creación manual de un archivo de copia de seguridad.'],
    ],
    'B' => [
        'B1' => ['action' => 'PASSWORD_CHANGED_SELF', 'desc' => 'Cambio de credenciales por parte del propio usuario (desde "Mi Perfil").'],
        'B2' => ['action' => 'FEEDBACK_SAVED', 'desc' => 'El paciente ha guardado una valoración (Dolor/Esfuerzo).'],
        'B3' => ['action' => 'FEEDBACK_DELETED', 'desc' => 'Eliminación de una valoración de progreso (Feedback).'],
        'B4' => ['action' => 'USER_REGISTERED', 'desc' => 'Alta de una nueva cuenta (Paciente).'],
        'B5' => ['action' => 'PASSWORD_RESET_REQUEST', 'desc' => 'Solicitud de reseteo de contraseña por email.'],
        'B6' => ['action' => 'PASSWORD_RESET_COMPLETED', 'desc' => 'Completado de reseteo de contraseña (vía email).'],
        'B7' => ['action' => 'USER_PROFILE_UPDATED', 'desc' => 'Modificación de datos de perfil (nombre, teléfono, avatar).'],
        'B8' => ['action' => 'USER_PREFS_UPDATED', 'desc' => 'Cambio de las preferencias de interfaz por defecto.'],
    ],
    'C' => [
        'C1' => ['action' => 'PATIENT_RECORD_VIEWED', 'desc' => 'Acceso a la ficha confidencial de un paciente.'],
        'C2' => ['action' => 'PATIENT_DELETED', 'desc' => 'CRÍTICO: Eliminación permanente de un paciente.'],
        'C3' => ['action' => 'PATIENT_MODIFIED', 'desc' => 'Modificación de datos personales de un paciente.'],
        'C4' => ['action' => 'PATIENT_ARCHIVED', 'desc' => 'Archivado/Desarchivado de un paciente (Dar de baja / Reactivar).'],
        'C5' => ['action' => 'TREATMENT_CREATED', 'desc' => 'Creación de un nuevo tratamiento (o clonación).'],
        'C6' => ['action' => 'TREATMENT_UPDATED', 'desc' => 'Modificación de un tratamiento existente (colaboradores, etc).'],
        'C7' => ['action' => 'TREATMENT_DELETED', 'desc' => 'CRÍTICO: Eliminación permanente de un tratamiento.'],
        'C8' => ['action' => 'TREATMENT_TRANSFERRED', 'desc' => 'CRÍTICO: Transferencia de propiedad de un tratamiento.'],
        'C9' => ['action' => 'EMAIL_SETTINGS_UPDATED', 'desc' => 'Modificación de la configuración de emails automáticos.'],
        'C10' => ['action' => 'BANNED_EMAIL_ADDED', 'desc' => 'Registro de un email en la lista de restricción de acceso (Baneo).'],
        'C11' => ['action' => 'BANNED_EMAIL_REMOVED', 'desc' => 'Eliminación de un email de la lista de restricción de acceso.'],
        'C12' => ['action' => 'PROHIBITED_PASS_ADDED', 'desc' => 'Restricción de una contraseña en la lista de prohibidas.'],
        'C13' => ['action' => 'PROHIBITED_PASS_REMOVED', 'desc' => 'Eliminación de una contraseña de la lista de prohibidas.'],
        'C14' => ['action' => 'CHAT_MESSAGE_SENT', 'desc' => 'Mensaje de Chat enviado (Texto, Imagen, o Quick Feedback).'],
        'C15' => ['action' => 'CHAT_CONV_CLEARED', 'desc' => 'Historial de Chat borrado permanentemente u ocultado de la vista del usuario.'],
    ]
];
$action_to_category = [];
foreach ($log_categories as $cat_key => $actions) {
    foreach ($actions as $code => $data) {
        $action_to_category[$data['action']] = $code;
    }
}
$categories_for_filter = array_keys($log_categories);

// --- Mapeo de Colores de TEXTO ---
$code_text_colors = [
    'A1' => 'text-danger',
    'A2' => 'text-secondary',
    'A3' => 'text-warning',
    'A4' => 'text-warning',
    'A5' => 'text-danger',
    'A6' => 'text-danger',
    'A7' => 'text-danger',
    'A8' => 'text-warning',
    'B1' => 'text-info',
    'B2' => 'text-success',
    'B3' => 'text-success',
    'B4' => 'text-primary',
    'B5' => 'text-info',
    'B6' => 'text-success',
    'B7' => 'text-primary',
    'B8' => 'text-info',
    'C1' => 'text-danger',
    'C2' => 'text-danger',
    'C3' => 'text-warning',
    'C4' => 'text-danger',
    'C5' => 'text-primary',
    'C6' => 'text-info',
    'C7' => 'text-danger',
    'C8' => 'text-warning',
    'C9' => 'text-info',
    'C10' => 'text-warning',
    'C11' => 'text-info',
    'C12' => 'text-secondary',
    'C13' => 'text-secondary',
    'C14' => 'text-primary',
    'C15' => 'text-info',
];

// --- 2. CONFIGURACIÓN DE PAGINACIÓN, FILTROS Y ORDENACIÓN ---
$records_per_page = 50;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $records_per_page;

// Parámetro de ordenación
$sort_order = strtoupper(trim($_GET['order'] ?? 'DESC'));
if (!in_array($sort_order, ['ASC', 'DESC'])) {
    $sort_order = 'DESC';
}

// --- 3. LECTURA DE FILTROS DESDE LA URL ---
$filter_actor_id = (int)($_GET['actor_id'] ?? 0);
$filter_action = trim($_GET['action'] ?? '');
$filter_date = trim($_GET['date'] ?? '');
$filter_ip = trim($_GET['ip'] ?? '');
$filter_category = trim($_GET['category'] ?? '');


// --- 4. PREPARACIÓN DE LA CONSULTA SQL ---
$sql_where_clauses = ["1=1"];
$sql_params = [];

// A) Filtro por Categoría
if (!empty($filter_category) && isset($log_categories[$filter_category])) {
    $actions_in_category = array_column($log_categories[$filter_category], 'action');
    $placeholders = implode(',', array_fill(0, count($actions_in_category), '?'));
    $sql_where_clauses[] = "action IN ({$placeholders})";
    $sql_params = array_merge($sql_params, $actions_in_category);
}

// B) Filtros Estándar
if ($filter_actor_id > 0) {
    $sql_where_clauses[] = "actor_user_id = ?";
    $sql_params[] = $filter_actor_id;
}
if (!empty($filter_action)) {
    $sql_where_clauses[] = "action = ?";
    $sql_params[] = $filter_action;
}
if (!empty($filter_ip)) {
    $sql_where_clauses[] = "ip_address LIKE ?";
    $sql_params[] = "%" . $filter_ip . "%";
}
if (!empty($filter_date)) {
    $sql_where_clauses[] = "DATE(timestamp) = ?";
    $sql_params[] = $filter_date;
}

$where_clause = " WHERE " . implode(" AND ", $sql_where_clauses);

try {
    // a) Contar el total de registros filtrados
    $stmt_count = $db->prepare("SELECT COUNT(*) FROM log_auditoria" . $where_clause);
    $stmt_count->execute($sql_params);
    $total_records = $stmt_count->fetchColumn();
    $total_pages = ceil($total_records / $records_per_page);

    // b) Obtener los datos filtrados, paginados y ordenados
    $sql_data = "
        SELECT * FROM log_auditoria
        " . $where_clause . "
        ORDER BY timestamp {$sort_order}
        LIMIT ? OFFSET ?
    ";

    $stmt_data = $db->prepare($sql_data);

    // (Paginación PDO)
    $param_index = 1;
    foreach ($sql_params as $value) {
        $stmt_data->bindValue($param_index, $value);
        $param_index++;
    }
    $stmt_data->bindValue($param_index, (int) $records_per_page, PDO::PARAM_INT);
    $param_index++;
    $stmt_data->bindValue($param_index, (int) $offset, PDO::PARAM_INT);
    $stmt_data->execute();
    $logs = $stmt_data->fetchAll(PDO::FETCH_ASSOC);


    // c) Obtener lista de actores y acciones únicas para los filtros
    $actors = $db->query("SELECT DISTINCT actor_user_id, actor_username FROM log_auditoria WHERE actor_user_id IS NOT NULL ORDER BY actor_username")->fetchAll(PDO::FETCH_ASSOC);
    $actions = $db->query("SELECT DISTINCT action FROM log_auditoria ORDER BY action")->fetchAll(PDO::FETCH_COLUMN);

} catch (Exception $e) {
    $error_message = "Error al consultar el registro: " . $e->getMessage();
    $logs = [];
    $total_records = 0;
    $total_pages = 1;
}

// Función auxiliar para mantener los filtros en la URL (usada en paginación y ordenación)
function get_filter_query($page, $order = null) {
    $query_params = $_GET;
    $current_order = $query_params['order'] ?? 'DESC';
    $query_params['page'] = $page;
    $query_params['order'] = $order ?? $current_order;
    if (isset($query_params['view'])) {
        $query_params['view'] = 'clean';
    }
    return http_build_query($query_params);
}

// --- RENDERIZADO HTML MÍNIMO ---
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($page_title) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">

    <style>
        /* Estilos para la vista limpia */
        body {
            padding-top: 20px !important;
            margin-top: 0 !important;
            min-height: 100vh;
        }
        /* Color de la cabecera de la tabla */
        .audit-table-container thead {
            background-color: #008080 !important; /* Teal Oscuro Forzado */
            color: white;
        }
        .audit-table-container thead a {
            color: white !important;
        }

        /* --- ★ INICIO MODIFICACIÓN: TAREA 1 (Tabla más densa) --- */
        .audit-table-container .table td,
        .audit-table-container .table th {
            font-size: 0.8rem; /* Reducimos el tamaño de fuente */
            padding: 0.4rem 0.5rem; /* Reducimos el padding (altura de fila) */
            vertical-align: middle;
        }
        /* --- ★ FIN MODIFICACIÓN --- */


        /* --- ★ INICIO MODIFICACIÓN: Estilos para impresión del modal ★ --- */
        @media print {
            /* Ocultar todo menos el modal cuando se imprime */
            body.modal-open > *:not(#codisModal) {
                display: none;
                visibility: hidden;
            }
            body.modal-open #codisModal {
                display: block;
                position: static;
            }
            body.modal-open #codisModal .modal-dialog {
                max-width: 100%;
                width: 100%;
                margin: 0;
            }
            body.modal-open #codisModal .modal-content {
                border: 0;
                box-shadow: none;
            }
            body.modal-open #codisModal .modal-header,
            body.modal-open #codisModal .modal-footer {
                /* Aseguramos que el encabezado y el pie de página desaparezcan al imprimir el modal */
                display: none !important;
            }
            body.modal-open #codisModal .no-print {
                display: none !important;
            }

            /* --- Estilos para el iframe en impresión --- */
            body.modal-open #codisModal iframe {
                width: 100%;
                height: 100vh; /* Ocupar toda la altura de la página de impresión */
            }
        }
        /* --- ★ FIN MODIFICACIÓN --- */

    </style>
</head>
<body class="d-flex flex-column h-100">

<main class="main-content container-fluid mt-0" style="max-width: 98%;">

    <div class="d-flex justify-content-between align-items-center mb-3">
        <h1 class="h3"><i class="bi bi-eye-fill me-2"></i>Registro de Auditoría del Sistema</h1>
        <div class="d-flex align-items-center">

            <button type="button" class="btn btn-sm btn-outline-info" data-bs-toggle="modal" data-bs-target="#codisModal" title="Consulta la tabla de códigos de auditoría">
                <i class="bi bi-book me-1"></i> Taula de Codis
            </button>
            <button type="button" class="btn btn-sm btn-outline-secondary ms-2" onclick="window.close();" data-bs-toggle="tooltip" title="Cerrar esta pestaña">
                <i class="bi bi-x-lg me-1"></i> Tancar
            </button>
        </div>
    </div>

    <?php if (isset($error_message)): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error_message) ?></div>
    <?php endif; ?>

    <div class="card shadow-sm mb-2">
        <div class="card-body">
            <form method="GET" action="admin_logs.php?view=clean" class="row g-3 align-items-end">
                <input type="hidden" name="order" value="<?= htmlspecialchars($sort_order) ?>">

                <div class="col-md-3">
                    <label for="filter_category" class="form-label small">Categoría</label>
                    <select id="filter_category" name="category" class="form-select form-select-sm">
                        <option value="">Todas las categorías</option>
                        <?php foreach ($categories_for_filter as $category): ?>
                            <option value="<?= htmlspecialchars($category) ?>" <?= $filter_category == $category ? 'selected' : '' ?>>
                                Categoría <?= $category ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="col-md-2">
                    <label for="filter_actor_id" class="form-label small">Actor (ID)</label>
                    <select id="filter_actor_id" name="actor_id" class="form-select form-select-sm">
                        <option value="0">Todos los actores</option>
                        <?php foreach ($actors as $actor): ?>
                            <option value="<?= $actor['actor_user_id'] ?>" <?= $filter_actor_id == $actor['actor_user_id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($actor['actor_username']) ?> (ID: <?= $actor['actor_user_id'] ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="col-md-2">
                    <label for="filter_action" class="form-label small">Acción Específica</label>
                    <select id="filter_action" name="action" class="form-select form-select-sm">
                        <option value="">Todas las acciones</option>
                        <?php foreach ($actions as $action): ?>
                            <option value="<?= htmlspecialchars($action) ?>" <?= $filter_action == $action ? 'selected' : '' ?>>
                                <?= htmlspecialchars($action) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="col-md-1">
                    <label for="filter_date" class="form-label small">Fecha</label>
                    <input type="date" id="filter_date" name="date" class="form-control form-control-sm" value="<?= htmlspecialchars($filter_date) ?>">
                </div>

                   <div class="col-md-2">
                    <label for="filter_ip" class="form-label small">IP de Origen</label>
                    <input type="text" id="filter_ip" name="ip" class="form-control form-control-sm" value="<?= htmlspecialchars($filter_ip) ?>" placeholder="192.168.1.x">
                </div>

                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary btn-sm w-100">
                        <i class="bi bi-funnel-fill me-1"></i> Filtrar
                    </button>
                    <a href="admin_logs.php?view=clean&order=<?= htmlspecialchars($sort_order) ?>" class="btn btn-outline-secondary btn-sm w-100 mt-2">
                        Limpiar Filtros
                    </a>
                </div>
            </form>
        </div>
    </div>

    <div class="d-flex justify-content-between align-items-center mb-2">
        <span class="badge bg-primary">Total Registros: <?= $total_records ?></span>

        </div>

    <div class="card shadow-sm">
        <div class="card-body p-0">
            <div class="table-responsive audit-table-container">
                <table class="table table-striped table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th scope="col" style="width: 5%;">Código</th>
                            <th scope="col" style="width: 4%;">#</th>
                            <th scope="col" style="width: 13%;">
                                <?php
                                $next_order = ($sort_order === 'DESC') ? 'ASC' : 'DESC';
                                $icon = ($sort_order === 'DESC') ? 'bi-sort-down-alt' : 'bi-sort-up';
                                ?>
                                <a href="?<?= get_filter_query(1, $next_order) ?>&view=clean" class="text-decoration-none d-flex align-items-center">
                                    Fecha/Hora
                                    <i class="bi <?= $icon ?> ms-2"></i>
                                </a>
                            </th>
                            <th scope="col" style="width: 15%;">Actor (Quién)</th>
                            <th scope="col" style="width: 15%;">Acción</th>
                            <th scope="col" style="width: 25%;">Detalles</th>
                            <th scope="col" style="width: 8%;">Objetivo (Usuario)</th>
                            <th scope="col" style="width: 8%;">Objetivo (ID)</th>
                            <th scope="col" style="width: 7%;">IP</th>
                            </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($logs) && $total_records === 0): ?>
                            <tr>
                                <td colspan="9" class="text-center text-muted p-4">No se han encontrado registros de auditoría.</td>
                                </tr>
                        <?php endif; ?>

                        <?php
                        $record_index = $offset;
                        foreach ($logs as $log):
                            $record_index++;

                            // --- LÓGICA PARA OBTENER CÓDIGO, DESCRIPCIÓN Y COLOR ---
                            $code = $action_to_category[$log['action']] ?? 'N/A';
                            $description = 'Acción desconocida';
                            $text_color_class = $code_text_colors[$code] ?? 'text-dark';

                            $category_key = substr($code, 0, 1);
                            if (isset($log_categories[$category_key]) && isset($log_categories[$category_key][$code])) {
                                $description = $log_categories[$category_key][$code]['desc'];
                            } else {
                                // Fallback para logs dinámicos o no categorizados
                                if (str_starts_with($log['action'], 'CHAT_')) {
                                    $description = 'Acción de mensajería: ' . $log['action'];
                                    if ($code === 'N/A') $code = 'C?';
                                } else if ($category_key === 'C') {
                                    $description = 'Acción clínica registrada: ' . $log['action'];
                                }
                            }
                        ?>
                            <tr>
                                <td class="text-center fw-bold"
                                    data-bs-toggle="tooltip" data-bs-placement="top"
                                    title="<?= htmlspecialchars($description) ?>"
                                    style="cursor: help;">
                                    <?= htmlspecialchars($code) ?>
                                </td>
                                <td class="text-muted small"><?= $record_index ?></td>
                                <td class="text-nowrap"><?= htmlspecialchars($log['timestamp']) ?></td>
                                <td>
                                    <?php if ($log['actor_username']): ?>
                                        <?= htmlspecialchars($log['actor_username']) ?> (ID: <?= htmlspecialchars($log['actor_user_id'] ?? 'N/A') ?>)
                                    <?php else: ?>
                                        <span class="text-muted">N/A</span>
                                    <?php endif; ?>
                                </td>
                                <td class="fw-bold">
                                    <?php
                                        $action = htmlspecialchars($log['action']);
                                        echo '<span class="' . $text_color_class . '">' . $action . '</span>';
                                    ?>
                                </td>

                                <td class="small text-muted text-truncate" style="max-width: 250px;"
                                    data-bs-toggle="tooltip" data-bs-placement="top" title="<?= htmlspecialchars($log['details'] ?? '--') ?>">
                                    <?= htmlspecialchars($log['details'] ?? '--') ?>
                                </td>
                                <td>
                                    <?php if ($log['target_user_id']): ?>
                                        ID Usuario: <span class="fw-bold"><?= htmlspecialchars($log['target_user_id']) ?></span>
                                    <?php else: ?>
                                        <span class="text-muted">--</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($log['target_object_id']): ?>
                                        Objeto/Detalle: <span class="fw-bold"><?= htmlspecialchars($log['target_object_id']) ?></span>
                                    <?php else: ?>
                                        <span class="text-muted">--</span>
                                    <?php endif; ?>
                                </td>
                                <td class="text-muted small text-nowrap"><?= htmlspecialchars($log['ip_address']) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <?php if ($total_pages > 1): ?>
        <nav aria-label="Paginación de Registro" class="mt-4">
            <ul class="pagination justify-content-center">

                <?php
                // Botón Anterior
                $prev_page = max(1, $current_page - 1);
                $disabled_prev = ($current_page <= 1) ? 'disabled' : '';
                echo "<li class='page-item {$disabled_prev}'><a class='page-link' href='?" . get_filter_query($prev_page) . "'>Anterior</a></li>";

                // Enlaces directos (Mostramos 5 páginas alrededor de la actual)
                $start_page = max(1, $current_page - 2);
                $end_page = min($total_pages, $current_page + 2);

                if ($start_page > 1) {
                    echo "<li class='page-item'><a class='page-link' href='?" . get_filter_query(1) . "'>1</a></li>";
                    if ($start_page > 2) {
                        echo "<li class='page-item disabled'><span class='page-link'>...</span></li>";
                    }
                }

                for ($i = $start_page; $i <= $end_page; $i++):
                    $active = ($i == $current_page) ? 'active' : '';
                    echo "<li class='page-item {$active}'><a class='page-link' href='?" . get_filter_query($i) . "'>{$i}</a></li>";
                endfor;

                if ($end_page < $total_pages) {
                    if ($end_page < $total_pages - 1) {
                        echo "<li class='page-item disabled'><span class='page-link'>...</span></li>";
                    }
                    echo "<li class='page-item'><a class='page-link' href='?" . get_filter_query($total_pages) . "'>{$total_pages}</a></li>";
                }

                // Botón Siguiente
                $next_page = min($total_pages, $current_page + 1);
                $disabled_next = ($current_page >= $total_pages) ? 'disabled' : '';
                echo "<li class='page-item {$disabled_next}'><a class='page-link' href='?" . get_filter_query($next_page) . "'>Siguiente</a></li>";
                ?>
            </ul>
        </nav>
        <div class="text-center small text-muted mt-2">
            Página <?= $current_page ?> de <?= $total_pages ?>
        </div>
    <?php endif; ?>
</main>

<div class="modal fade" id="codisModal" tabindex="-1" aria-labelledby="codisModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-scrollable">
      <div class="modal-content">
          <div class="modal-header">
              <h5 class="modal-title" id="codisModalLabel">Registro de Auditoría: Clasificación y Prioridades</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">

              <iframe src="help/table_logs.html" style="width: 100%; height: 65vh; border: none;" title="Tabla de clasificación de logs">
                  Tu navegador no soporta iframes. Por favor, accede a <a href="table_logs.html">table_logs.html</a> directamente.
              </iframe>
          </div>
          <div class="modal-footer">
              <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
          </div>
      </div>
  </div>
</div>
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Inicialización de Tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
      return new bootstrap.Tooltip(tooltipTriggerEl);
    });
});
</script>
</body>
</html>
