<?php
// /reset_password.php
session_start();
require 'db.php';
require_once 'logger.php'; // <-- Delegación de logs (B6)

$error = '';
$token = $_GET['token'] ?? null;
$token_valido = false;
$user_email = null;
$user_id = null; // Inicializamos para almacenar el ID del usuario

if (empty($token)) {
    $error = "Token invàlid o inexistent.";
} else {
    // 1. Validar el token: que exista Y que no haya caducado
    $stmt = $db->prepare("SELECT * FROM password_resets WHERE token = ? AND expires_at > ?");
    $stmt->execute([$token, date('Y-m-d H:i:s')]);
    $reset_request = $stmt->fetch();

    if ($reset_request) {
        $token_valido = true;
        $user_email = $reset_request['email'];

        // Obtenemos el ID de usuario para el log y la actualización de contraseña
        $stmt_user = $db->prepare("SELECT id FROM cuentas WHERE email = ?");
        $stmt_user->execute([$user_email]);
        $user_data = $stmt_user->fetch(PDO::FETCH_ASSOC);

        if ($user_data) {
            $user_id = $user_data['id']; // ID obtenido
        } else {
            // El token es válido, pero el usuario ya no existe o fue eliminado
            $error = "L'enllaç de recuperació és invàlid o ha caducat. Si us plau, sol·licita'n un de nou.";
        }
    } else {
        $error = "L'enllaç de recuperació és invàlid o ha caducat. Si us plau, sol·licita'n un de nou.";
    }
}

// 2. Procesar el formulario de nueva contraseña
if ($token_valido && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $password = $_POST['password'];
    $password_confirm = $_POST['password_confirm'];

    // Validación según la lógica proporcionada
    if (empty($password) || empty($password_confirm)) {
        $error = "Els camps de contrasenya no poden estar buits.";
    } elseif (strlen($password) < 6) { // Validación mínima de 6 caracteres
        $error = "La contrasenya ha de tenir almenys 6 caràcters.";
    } elseif (!preg_match('/^(?=.*\d)(?=.*[A-Z]).{6,12}$/', $password)) {
        // Añadimos la validación completa que se usa en otros archivos (min 6-12, Mayús, Num)
        $error = 'La contrasenya ha de tindre entre 6 i 12 caràcters, i incloure almenys una lletra majúscula i un número.';
    } elseif ($password !== $password_confirm) {
        $error = "Les contrasenyes no coincideixen.";
    }

    if (empty($error)) {
        // Aseguramos que tenemos el ID antes de proceder
        if ($user_id && $user_email) {
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);

            // Transacción para garantizar atomicidad
            $db->beginTransaction();
            try {
                // 3. Actualizar la contraseña
                $stmt_update = $db->prepare("UPDATE cuentas SET password = ? WHERE id = ?");
                $stmt_update->execute([$hashed_password, $user_id]);

                // 4. Eliminar el token
                $stmt_delete = $db->prepare("DELETE FROM password_resets WHERE email = ?");
                $stmt_delete->execute([$user_email]);

                // ★★★ REGISTRO DE AUDITORÍA DELEGADO (B6: PASSWORD_RESET_COMPLETED) ★★★
                // Delegamos la construcción del log al helper
                registrar_reset_completado($db, $user_id, $user_email);
                // ★★★ FIN REGISTRO DE AUDITORÍA ★★★

                $db->commit();

                // 5. Redirigir al login con mensaje de éxito
                $_SESSION['success_message'] = "Contrasenya actualitzada correctament. Ja pots iniciar sessió.";
                header('Location: index.php');
                exit;

            } catch (Exception $e) {
                $db->rollBack();
                error_log("Error al completar reseteo de contraseña para " . $user_email . ": " . $e->getMessage());
                $error = "Hi ha hagut un error en actualitzar la contrasenya. Si us plau, intenta-ho de nou.";
            }

        } else {
            $error = "Error intern: No s'ha pogut trobar el compte.";
        }
    }
}
$page_title = "Restablir Contrasenya";
?>
<!DOCTYPE html>
<html lang="ca">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
    <link rel="stylesheet" href="style.css">
    <style>
        body { background-color: #e9ecef; }
        /* Estilos para el botón de mostrar/ocultar contraseña */
        .password-group { position: relative; }
        .password-toggle-btn { position: absolute; right: 10px; top: 50%; transform: translateY(-50%); background: none; border: none; }
    </style>
</head>
<body>
<main class="container py-5">
    <div class="row justify-content-center">
        <div class="col-md-6 col-lg-4">
            <div class="text-center mb-4">
                 <img src="logos/logo2.jpg" alt="Logotip d'AVANT ONLINE" class="mb-3 logo-circle">
            </div>
            <div class="card shadow">
                <div class="card-body p-4">
                    <h3 class="text-center mb-4">Crear Nova Contrasenya</h3>

                    <?php if ($error): ?><div class="alert alert-danger py-2"><?= htmlspecialchars($error) ?></div><?php endif; ?>

                    <?php if ($token_valido): ?>
                    <form method="POST" action="reset_password.php?token=<?= htmlspecialchars($token) ?>">
                        <div class="mb-3">
                            <label for="password" class="form-label">Nova Contrasenya</label>
                            <div class="password-group">
                                <input type="password" class="form-control" id="password" name="password" required>
                                <button type="button" class="password-toggle-btn"><i class="bi bi-eye-slash"></i></button>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label for="password_confirm" class="form-label">Confirma la Contrasenya</label>
                            <div class="password-group">
                                <input type="password" class="form-control" id="password_confirm" name="password_confirm" required>
                                <button type="button" class="password-toggle-btn"><i class="bi bi-eye-slash"></i></button>
                            </div>
                        </div>
                        <button type="submit" class="btn btn-primary w-100 mt-3">Guardar Contrasenya</button>
                    </form>
                    <?php else: ?>
                        <div class="text-center">
                            <a href="index.php" class="btn btn-secondary">Tornar a l'inici</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</main>
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
// Script para mostrar/ocultar contraseña (funcionalidad copiada)
$(document).ready(function() {
    $(document).on('click', '.password-toggle-btn', function() {
        const passwordInput = $(this).closest('.password-group').find('input');
        const icon = $(this).find('i');
        if (passwordInput.attr('type') === 'password') {
            passwordInput.attr('type', 'text');
            icon.removeClass('bi-eye-slash').addClass('bi-eye');
        } else {
            passwordInput.attr('type', 'password');
            icon.removeClass('bi-eye').addClass('bi-eye-slash');
        }
    });
});
</script>
</body>
</html>
