<?php
// /mail_helpers.php
// Función centralizada para enviar emails.
require_once 'vendor/autoload.php'; // Requiere Composer para PHPMailer (el método más seguro)
require_once 'config_smtp.php'; // Incluir la configuración

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/**
 * Función genérica para enviar correos electrónicos a un destinatario.
 *
 * @param string $toEmail
 * @param string $subject
 * @param string $bodyHtml
 * @param bool $isHtml
 * @param array $embedded_images Array de imágenes para incrustar.
 * @param array $bcc_list Lista de emails en copia oculta (Bcc).
 * @param string|null $settingName Clave de configuración de la tabla email_settings (ej: 'EMAIL_USER_REGISTERED').
 */
function send_platform_email($toEmail, $subject, $bodyHtml, $isHtml = true, $embedded_images = [], $bcc_list = [], $settingName = null) {

    // --- ★★★ INICIO: OBTENER CONEXIÓN DB GLOBAL ★★★ ---
    // Esta función es llamada por archivos que ya han incluido 'db.php',
    // por lo que $db está disponible en el ámbito global.
    global $db;
    // --- ★★★ FIN: OBTENER CONEXIÓN DB GLOBAL ★★★ ---


    // --- ★★★ INICIO: COMPROBACIÓN CENTRALIZADA DE ACTIVACIÓN (Paso B) ★★★ ---
    // La función is_email_setting_enabled() se define en db.php, que se asume incluido en el punto de llamada.

    // --- MODIFICACIÓN: Esta lógica está rota porque 'is_email_setting_enabled' NO existe en db.php ---
    // --- La lógica de comprobación manual se debe hacer ANTES de llamar a esta función ---
    /*
    if ($settingName && function_exists('is_email_setting_enabled')) {
        if (!is_email_setting_enabled($settingName)) {
            // El email está deshabilitado por el administrador. Se registra y se sale.
            error_log("EMAIL SKIPPED: El tipus de comunicació '$settingName' està deshabilitat per l'administrador.");
            return true; // Procesado (no enviado intencionadamente)
        }
    }
    */
    // --- ★★★ FIN: COMPROBACIÓN CENTRALIZADA ★★★ ---

    $mail = new PHPMailer(true); // Pasar 'true' habilita las excepciones

    try {
        // Configuración del servidor (Paso 1)
        $mail->isSMTP();
        $mail->Host       = SMTP_HOST;
        $mail->SMTPAuth   = true;
        $mail->Username   = SMTP_USER;
        $mail->Password   = SMTP_PASS;
        $mail->SMTPSecure = SMTP_SECURE;
        $mail->Port       = SMTP_PORT;

        // ★ Setejar el CharSet correctament per accents (important)
        $mail->CharSet = 'UTF-8';

        // Remitente y Destinatario
        $mail->setFrom(MAIL_FROM_EMAIL, MAIL_FROM_NAME);
        $mail->addAddress($toEmail);

        // Copias ocultas (Bcc)
        if (!empty($bcc_list)) {
            foreach ($bcc_list as $bccEmail) {
                $mail->addBCC($bccEmail);
            }
        }

        // Contenido
        $mail->isHTML($isHtml);
        $mail->Subject = $subject;
        $mail->Body    = $bodyHtml;
        $mail->AltBody = strip_tags($bodyHtml); // Versión de texto plano

        // --- ★ INICI: MODIFICACIÓ PER INCRUSTAR IMATGES ★ ---
        if (!empty($embedded_images)) {
            foreach ($embedded_images as $image) {
                if (isset($image['path']) && isset($image['cid'])) {
                    if (file_exists($image['path'])) {
                        $mail->addEmbeddedImage($image['path'], $image['cid']);
                    } else {
                        error_log("Error al incrustar imatge: No es troba el fitxer " . $image['path']);
                    }
                }
            }
        }
        // --- ★ FI: MODIFICACIÓ PER INCRUSTAR IMATGES ★ ---

        $mail->send();
        return true;

    } catch (Exception $e) {
        // --- ======================================================= ---
        // --- ★★★ INICIO: MODIFICACIÓN (NOTIFICAR FALLO AL ADMIN) ★★★ ---
        // --- ======================================================= ---

        // 1. Registrar el error original (como antes)
        $original_error_message = "Error al enviar correo a $toEmail. Mailer Error: {$mail->ErrorInfo}";
        error_log($original_error_message);

        // 2. Intentar notificar a los Superadmins
        if ($db) {
            try {
                // 3. Buscar los emails de los superadmins activos (lógica de register.php)
                $stmt_admins = $db->query("SELECT email FROM cuentas WHERE rol = 'superadmin' AND status = 'activo'");
                $admin_emails = $stmt_admins->fetchAll(PDO::FETCH_COLUMN);

                if (!empty($admin_emails)) {
                    $first_admin = array_shift($admin_emails);
                    $bcc_list_admins = $admin_emails;

                    $subject_admin = "Alerta del Sistema: Ha fallat un enviament d'email";
                    $body_admin = "
                        <div style='font-family: Arial, sans-serif; padding: 15px; border: 1px solid #ddd;'>
                        <h2 style='color: #D9534F;'>Error Crític en l'Enviament d'Email</h2>
                        <p>La plataforma ha intentat enviar un email que ha fallat. Això pot ser degut a una adreça de correu incorrecta o un error del servidor.</p>
                        <ul style='background-color: #f9f9f9; padding: 10px;'>
                            <li><strong>Destinatari que ha fallat:</strong> " . htmlspecialchars($toEmail) . "</li>
                            <li><strong>Assumpte:</strong> " . htmlspecialchars($subject) . "</li>
                            <li><strong>Error Detallat:</strong> <pre>" . htmlspecialchars($mail->ErrorInfo) . "</pre></li>
                        </ul>
                        <p style='font-size: 0.9em; color: #777;'>Aquest és un missatge automàtic d'alerta del sistema.</p>
                        </div>
                    ";

                    // 4. Crear una instancia NUEVA de PHPMailer para la alerta
                    $alertMail = new PHPMailer(false); // 'false' para no lanzar excepciones aquí

                    // 5. Configurar y enviar la alerta (usando las mismas constantes SMTP)
                    $alertMail->isSMTP();
                    $alertMail->Host       = SMTP_HOST;
                    $alertMail->SMTPAuth   = true;
                    $alertMail->Username   = SMTP_USER;
                    $alertMail->Password   = SMTP_PASS;
                    $alertMail->SMTPSecure = SMTP_SECURE;
                    $alertMail->Port       = SMTP_PORT;
                    $alertMail->CharSet    = 'UTF-8';

                    $alertMail->setFrom(MAIL_FROM_EMAIL, "Sistema AVANT ONLINE");
                    $alertMail->addAddress($first_admin);
                    foreach ($bcc_list_admins as $bcc) {
                        $alertMail->addBCC($bcc);
                    }

                    $alertMail->isHTML(true);
                    $alertMail->Subject = $subject_admin;
                    $alertMail->Body    = $body_admin;

                    $alertMail->send();
                }
            } catch (Exception $alertException) {
                // Si la alerta al admin también falla, solo lo registramos
                error_log("FALLO CRÍTICO: No s'ha pogut enviar l'alerta d'email fallit als admins. Error: " . $alertException->getMessage());
            }
        } else {
            error_log("FALLO CRÍTICO: \$db no disponible. No s'ha pogut notificar als admins de l'error d'email.");
        }

        // 6. Devolver 'false' como antes
        return false;
        // --- ======================================================= ---
        // --- ★★★ FIN: MODIFICACIÓN ★★★ ---
        // --- ======================================================= ---
    }
}

?>
