<?php
// /forgot_password.php
session_start();
require 'db.php';
require 'mail_helpers.php'; // Incluimos el helper de email
require_once 'logger.php'; // <-- ★★★ DELEGACIÓN: Centralización de logs (B5) ★★★

$message = '';
$error = '';
$page_title = "Recuperar Contrasenya";

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email']);

    if (empty($email)) {
        $error = 'Per favor, introdueix el teu email.';
    } else {
        $stmt = $db->prepare("SELECT id, nombre, email FROM cuentas WHERE email = ? AND is_archived = 0");
        $stmt->execute([$email]);
        $user = $stmt->fetch();

        // IMPORTANTE: Por seguridad, siempre mostramos el mismo mensaje
        // tanto si el email existe como si no.
        $message = "Si existeix un compte amb aquest email, s'hi ha enviat un enllaç de recuperació.";

        if ($user) {
            try {
                // 1. Generar token seguro
                $token = bin2hex(random_bytes(32));
                // 2. Definir expiración (1 hora)
                $expires = date('Y-m-d H:i:s', time() + 3600);

                // 3. Guardar token en la BBDD (usando la tabla 'password_resets')
                // Eliminamos tokens anteriores para este email antes de insertar el nuevo
                $db->prepare("DELETE FROM password_resets WHERE email = ?")->execute([$user['email']]);
                $stmt_insert = $db->prepare("INSERT INTO password_resets (email, token, expires_at) VALUES (?, ?, ?)");
                $stmt_insert->execute([$user['email'], $token, $expires]);

                // --- ======================================================= ---
                // --- ★★★ INICI DE LA CORRECCIÓ: URL BASE DINÀMICA ★★★ ---
                // --- ======================================================= ---
                $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://";
                $domain = $_SERVER['HTTP_HOST'];
                // Detectar carpeta actual i afegir barra final
                // NOTA: Amb dirname($_SERVER['SCRIPT_NAME']), s'obté la carpeta, com a '/my/avant-online'
                $path = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
                $base_url = $protocol . $domain . $path . '/';
                // --- ★★★ FI DE LA CORRECCIÓ ★★★ ---
                // --- ======================================================= ---

                // 4. Crear el enlace de reseteo (USANT LA NOVA $base_url)
                // !!! ⚠️ ELIMINADA URL CODIFICADA: 'http://localhost/my/avant-online/reset_password.php?token=' !!!
                $reset_link = $base_url . "reset_password.php?token=" . $token;


                // --- ======================================================= ---
                // --- ★★★ INICIO DE LA CORRECCIÓN (INTERRUPTOR MANUAL) ★★★ ---
                // --- ======================================================= ---

                // 1. Comprobar el interruptor ESPECÍFICO para 'password_reset'
                $stmt_check = $db->prepare("SELECT is_enabled FROM email_settings WHERE setting_name = 'password_reset' LIMIT 1");
                $stmt_check->execute();

                // 2. Si la consulta devuelve '1' (ON), enviar el correo.
                if ($stmt_check->fetchColumn() == 1) {

                    // 5. Definir la imagen
                    $logo_path = __DIR__ . '/logos/img_defecte.png'; // Usamos la imagen correcta
                    $embedded_logo = [
                        ['path' => $logo_path, 'cid' => 'logo_avant']
                    ];

                    // 6. Preparar el email (usando la nueva plantilla HTML)
                    $subject = "Recuperar contrasenya - AVANT ONLINE";
                    $bodyHtml = "
                    <div style='font-family: Arial, sans-serif; line-height: 1.6; max-width: 600px; margin: auto; padding: 20px; border: 1px solid #ddd; border-radius: 8px;'>
                        <div style='text-align: center; margin-bottom: 20px;'>
                            <img src='cid:logo_avant' alt='AVANT ONLINE' style='width: 250px; height: auto;'>
                        </div>
                        <h1 style='color: #333; text-align: center; font-size: 24px;'>Hola, " . htmlspecialchars($user['nombre']) . "!</h1>
                        <p>Hem rebut una sol·licitud per restablir la contrasenya del teu compte a AVANT ONLINE.</p>
                        <p>Si no has sigut tu, pots ignorar aquest correu.</p>
                        <p>Per crear una nova contrasenya, fes clic al botó següent (l'enllaç caduca en 1 hora):</p>
                        <p style='text-align: center; margin-top: 30px;'>
                            <a href='" . $reset_link . "' style='background-color: #0d6efd; color: white; padding: 12px 20px; text-decoration: none; border-radius: 5px; font-weight: bold;'>
                                Restablir Contrasenya
                            </a>
                        </p>
                        <p style='color: #777; font-size: 0.9em; text-align: center; margin-top: 20px;'>
                            Si el botó no funciona, copia i enganxa aquesta URL al navegador:<br>
                            <span style='font-size: 0.8em; color: #999;'>" . $reset_link . "</span>
                        </p>
                    </div>
                    ";

                    if (send_platform_email($user['email'], $subject, $bodyHtml, true, $embedded_logo)) {
                        // ★★★ INICIO LOG B5 (DELEGADO) ★★★
                        // Antes: registrar_log($db, 'PASSWORD_RESET_REQUEST', $user['id'], $user['nombre'], $log_details_req, $user['id']);
                        registrar_reset_solicitado($db, $user['email'], $user['id']);
                        // ★★★ FIN LOG B5 ★★★
                    } else {
                        // Error de envío (el mensaje al usuario sigue siendo el mismo por seguridad)
                        error_log("Error al enviar email de reseteo a " . $user['email']);
                    }

                } else {
                    // El interruptor está en 0, no hacemos nada, solo log
                    error_log("EMAIL SKIPPED: El switch 'password_reset' está deshabilitado.");
                }
                // --- ======================================================= ---
                // --- ★★★ FIN DE LA CORRECCIÓN ★★★ ---
                // --- ======================================================= ---

            } catch (Exception $e) {
                error_log("Error al procesar reseteo de contraseña: " . $e->getMessage());
                // No mostramos el error al usuario por seguridad
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ca">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
    <link rel="stylesheet" href="style.css">
    <style>
        body { background-color: #e9ecef; }
    </style>
</head>
<body>
<main class="container py-5">
    <div class="row justify-content-center">
        <div class="col-md-6 col-lg-4">
            <div class="text-center mb-4">
                <img src="logos/logo2.jpg" alt="Logotip d'AVANT ONLINE" class="mb-3 logo-circle">
                <h1 class="h3 fw-bold" style="color: var(--custom-brand-purple);">
                    <i class="bi bi-heart-pulse-fill me-2"></i>Avant Online
                </h1>
            </div>
            <div class="card shadow">
                <div class="card-body p-4">
                    <h3 class="text-center mb-4">Recuperar Contrasenya</h3>

                    <?php if ($message): ?><div class="alert alert-info py-2"><?= htmlspecialchars($message) ?></div><?php endif; ?>
                    <?php if ($error): ?><div class="alert alert-danger py-2"><?= htmlspecialchars($error) ?></div><?php endif; ?>

                    <?php if (empty($message)): // Ocultar formulario si ya se envió ?>
                    <p class="text-muted small mb-3">Introdueix el teu email i t'enviarem un enllaç per restablir la teva contrasenya.</p>
                    <form method="POST" action="forgot_password.php">
                        <div class="mb-3">
                            <label for="email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="email" name="email" required>
                        </div>
                        <button type="submit" class="btn btn-primary w-100 mt-3">Enviar Enllaç</button>
                    </form>
                    <?php endif; ?>

                    <div class="text-center mt-4">
                        <a href="index.php" class="small">Tornar a l'inici de sessió</a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</main>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
